<?php
/**
 * WPG Admin — WordPress admin integration.
 *
 * Registers the admin menu, enqueues assets, and renders dashboard views.
 *
 * @package aos-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

class WPG_Admin
{

    /** @var WPG_Admin|null */
    private static ?WPG_Admin $instance = null;

    public static function instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_filter('plugin_action_links_' . WPG_PLUGIN_BASENAME, [$this, 'action_links']);

        // Show welcome notice on first activation.
        add_action('admin_notices', [$this, 'activation_notice']);
    }

    /**
     * Register the admin menu.
     */
    public function register_menu(): void
    {
        // Main menu.
        add_menu_page(
            __('AOS Governance', 'aos-governance'),
            __('AOS Governance', 'aos-governance'),
            'manage_options',
            'wpg-dashboard',
        [$this, 'render_dashboard'],
            'dashicons-shield',
            30
        );

        // Submenu pages.
        add_submenu_page(
            'wpg-dashboard',
            __('Dashboard', 'aos-governance'),
            __('Dashboard', 'aos-governance'),
            'manage_options',
            'wpg-dashboard',
        [$this, 'render_dashboard']
        );

        add_submenu_page(
            'wpg-dashboard',
            __('Policies', 'aos-governance'),
            __('Policies', 'aos-governance'),
            'manage_options',
            'wpg-policies',
        [$this, 'render_policies']
        );

        add_submenu_page(
            'wpg-dashboard',
            __('Audit Log', 'aos-governance'),
            __('Audit Log', 'aos-governance'),
            'manage_options',
            'wpg-audit-log',
        [$this, 'render_audit_log']
        );

        add_submenu_page(
            'wpg-dashboard',
            __('Documentation', 'aos-governance'),
            __('Documentation', 'aos-governance'),
            'manage_options',
            'wpg-docs',
        [$this, 'render_docs']
        );

        add_submenu_page(
            'wpg-dashboard',
            __('Settings', 'aos-governance'),
            __('Settings', 'aos-governance'),
            'manage_options',
            'wpg-settings',
        [$this, 'render_settings']
        );

        // Pro upsell tab with orange highlight.
        add_submenu_page(
            'wpg-dashboard',
            __('Go Pro', 'aos-governance'),
            '<span style="color: #f59e0b;">⭐ Go Pro</span>',
            'manage_options',
            'wpg-pro',
        [$this, 'render_pro']
        );
    }

    /**
     * Enqueue admin assets.
     *
     * @param string $hook_suffix The current admin page hook.
     */
    public function enqueue_assets(string $hook_suffix): void
    {
        // Only load on our pages.
        if (!str_contains($hook_suffix, 'wpg-')) {
            return;
        }

        wp_enqueue_style(
            'wpg-admin',
            WPG_PLUGIN_URL . 'admin/css/wpg-admin.css',
        [],
            WPG_VERSION
        );

        wp_enqueue_script(
            'wpg-admin',
            WPG_PLUGIN_URL . 'admin/js/wpg-admin.js',
        ['wp-api-fetch'],
            WPG_VERSION,
            true
        );

        wp_localize_script('wpg-admin', 'wpgAdmin', [
            'restUrl' => rest_url('wpg/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'version' => WPG_VERSION,
            'adminUrl' => admin_url('admin.php'),
        ]);
    }

    /**
     * Add plugin action links.
     *
     * @param array $links Existing plugin action links.
     *
     * @return array Modified links.
     */
    public function action_links(array $links): array
    {
        $custom_links = [
            '<a href="' . admin_url('admin.php?page=wpg-dashboard') . '">' . __('Dashboard', 'aos-governance') . '</a>',
            '<a href="' . admin_url('admin.php?page=wpg-policies') . '">' . __('Policies', 'aos-governance') . '</a>',
            '<a href="' . admin_url('admin.php?page=wpg-pro') . '" style="color: #e85d04; font-weight: 700;">' . __('Go Pro ⭐', 'aos-governance') . '</a>',
        ];

        return array_merge($custom_links, $links);
    }

    /**
     * Show welcome notice after activation.
     */
    public function activation_notice(): void
    {
        if (!get_transient('wpg_activated')) {
            return;
        }

        delete_transient('wpg_activated');

        printf(
            '<div class="notice notice-success is-dismissible"><p>%s <a href="%s">%s</a></p></div>',
            esc_html__('🛡️ AOS Governance for WP is active! Your constitutional AI governance layer is now protecting this site.', 'aos-governance'),
            esc_url(admin_url('admin.php?page=wpg-dashboard')),
            esc_html__('View Dashboard →', 'aos-governance')
        );
    }

    // ===== Render Methods =====

    public function render_dashboard(): void
    {
        include WPG_PLUGIN_DIR . 'admin/views/dashboard.php';
    }

    public function render_policies(): void
    {
        include WPG_PLUGIN_DIR . 'admin/views/policies.php';
    }

    public function render_audit_log(): void
    {
        include WPG_PLUGIN_DIR . 'admin/views/audit-log.php';
    }

    public function render_docs(): void
    {
        include WPG_PLUGIN_DIR . 'admin/views/documentation.php';
    }

    public function render_settings(): void
    {
        include WPG_PLUGIN_DIR . 'admin/views/settings.php';
    }

    public function render_pro(): void
    {
        include WPG_PLUGIN_DIR . 'admin/views/pro.php';
    }
}
