<?php
/**
 * WPG Abilities — Registers AOS governance abilities as MCP tools.
 *
 * This lets AI agents natively interact with the governance layer:
 * - wpg/check-policy: Pre-flight policy check
 * - wpg/audit-log: Query the audit trail
 * - wpg/active-policies: List active policies
 * - wpg/agent-profile: View the agent's own permissions
 * - wpg/report-generate: Generate compliance reports
 *
 * @package aos-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

class WPG_Abilities
{

    /** @var WPG_Abilities|null */
    private static ?WPG_Abilities $instance = null;

    public static function instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('wp_abilities_api_init', [$this, 'register_abilities']);
        add_action('wp_abilities_api_categories_init', [$this, 'register_category']);
    }

    /**
     * Register the AOS Governance ability category.
     */
    public function register_category(): void
    {
        if (!function_exists('wp_register_ability_category')) {
            return;
        }

        wp_register_ability_category('wpg-governance', [
            'label' => __('AOS Governance for WP', 'aos-governance'),
            'description' => __('Constitutional AI governance tools for policy enforcement, audit logging, and compliance.', 'aos-governance'),
        ]);
    }

    /**
     * Register all AOS governance abilities.
     */
    public function register_abilities(): void
    {
        if (!function_exists('wp_register_ability')) {
            return;
        }

        // --- wpg/check-policy ---
        wp_register_ability('wpg/check-policy', [
            'label' => __('Check Policy', 'aos-governance'),
            'description' => __('Pre-flight check: evaluate whether an ability action is allowed by current governance policies without executing it.', 'aos-governance'),
            'category' => 'wpg-governance',
            'input_schema' => [
                'type' => 'object',
                'properties' => [
                    'ability' => [
                        'type' => 'string',
                        'description' => 'The ability name to check (e.g., "core/delete-posts").',
                        'required' => true,
                    ],
                    'args' => [
                        'type' => 'object',
                        'description' => 'The arguments that would be passed to the ability.',
                        'default' => new \stdClass(),
                    ],
                ],
                'required' => ['ability'],
            ],
            'output_schema' => [
                'type' => 'object',
                'properties' => [
                    'decision' => ['type' => 'string', 'enum' => ['allow', 'deny']],
                    'policy_id' => ['type' => 'string'],
                    'policy_name' => ['type' => 'string'],
                    'reason' => ['type' => 'string'],
                    'elapsed_ms' => ['type' => 'number'],
                ],
            ],
            'permission_callback' => function () {
            return current_user_can('read');
        },
            'callback' => [$this, 'handle_check_policy'],
            'meta' => [
                'mcp' => ['type' => 'tool', 'public' => true],
            ],
        ]);

        // --- wpg/audit-log ---
        wp_register_ability('wpg/audit-log', [
            'label' => __('Audit Log', 'aos-governance'),
            'description' => __('Query the immutable governance audit trail with filters for decision, ability, agent, date range, and more.', 'aos-governance'),
            'category' => 'wpg-governance',
            'input_schema' => [
                'type' => 'object',
                'properties' => [
                    'decision' => [
                        'type' => 'string',
                        'enum' => ['allow', 'deny'],
                        'description' => 'Filter by decision type.',
                    ],
                    'ability' => [
                        'type' => 'string',
                        'description' => 'Filter by ability name (supports * wildcards).',
                    ],
                    'agent' => [
                        'type' => 'string',
                        'description' => 'Filter by agent identifier.',
                    ],
                    'from' => [
                        'type' => 'string',
                        'description' => 'Start date in Y-m-d H:i:s format.',
                    ],
                    'to' => [
                        'type' => 'string',
                        'description' => 'End date in Y-m-d H:i:s format.',
                    ],
                    'per_page' => [
                        'type' => 'integer',
                        'description' => 'Results per page (max 200).',
                        'default' => 20,
                    ],
                    'page' => [
                        'type' => 'integer',
                        'description' => 'Page number.',
                        'default' => 1,
                    ],
                ],
            ],
            'permission_callback' => function () {
            return current_user_can('manage_options');
        },
            'callback' => [$this, 'handle_audit_log'],
            'meta' => [
                'mcp' => ['type' => 'resource', 'public' => false],
            ],
        ]);

        // --- wpg/active-policies ---
        wp_register_ability('wpg/active-policies', [
            'label' => __('Active Policies', 'aos-governance'),
            'description' => __('List all active governance policies and their rules. Helps agents understand what boundaries are in place.', 'aos-governance'),
            'category' => 'wpg-governance',
            'input_schema' => [
                'type' => 'object',
                'properties' => new \stdClass(),
            ],
            'permission_callback' => function () {
            return current_user_can('read');
        },
            'callback' => [$this, 'handle_active_policies'],
            'meta' => [
                'mcp' => ['type' => 'resource', 'public' => true],
            ],
        ]);

        // --- wpg/agent-profile ---
        wp_register_ability('wpg/agent-profile', [
            'label' => __('Agent Profile', 'aos-governance'),
            'description' => __('View the current agent\'s detected identity, permissions, and restrictions.', 'aos-governance'),
            'category' => 'wpg-governance',
            'input_schema' => [
                'type' => 'object',
                'properties' => new \stdClass(),
            ],
            'permission_callback' => function () {
            return current_user_can('read');
        },
            'callback' => [$this, 'handle_agent_profile'],
            'meta' => [
                'mcp' => ['type' => 'resource', 'public' => true],
            ],
        ]);

        // --- wpg/report-generate ---
        wp_register_ability('wpg/report-generate', [
            'label' => __('Generate Compliance Report', 'aos-governance'),
            'description' => __('Generate a governance compliance report for a specified date range. Useful for HIPAA, SOC2, GDPR audits.', 'aos-governance'),
            'category' => 'wpg-governance',
            'input_schema' => [
                'type' => 'object',
                'properties' => [
                    'period' => [
                        'type' => 'string',
                        'enum' => ['day', 'week', 'month', 'all'],
                        'description' => 'The time period for the report.',
                        'default' => 'week',
                    ],
                ],
            ],
            'permission_callback' => function () {
            return current_user_can('manage_options');
        },
            'callback' => [$this, 'handle_report_generate'],
            'meta' => [
                'mcp' => ['type' => 'tool', 'public' => false],
            ],
        ]);
    }

    /**
     * Handle wpg/check-policy ability call.
     *
     * @param array $input The input data.
     *
     * @return array The evaluation result.
     */
    public function handle_check_policy(array $input): array
    {
        $agent_info = WPG_Agent_Detector::instance()->detect();

        $context = [
            'ability' => sanitize_text_field($input['ability'] ?? ''),
            'args' => $input['args'] ?? [],
            'agent' => $agent_info['agent'],
            'agent_type' => $agent_info['agent_type'],
            'user_id' => get_current_user_id(),
            'session_id' => $agent_info['session_id'],
        ];

        // Temporarily pause the gate so this check doesn't get gated itself.
        WPG_Gate::instance()->pause();
        $result = WPG_Policy_Engine::instance()->evaluate($context);
        WPG_Gate::instance()->resume();

        return $result->to_array();
    }

    /**
     * Handle wpg/audit-log ability call.
     *
     * @param array $input The query filters.
     *
     * @return array The audit log entries.
     */
    public function handle_audit_log(array $input): array
    {
        return WPG_Audit_Log::instance()->query($input);
    }

    /**
     * Handle wpg/active-policies ability call.
     *
     * @param array $input Unused.
     *
     * @return array The active policies.
     */
    public function handle_active_policies(array $input): array
    {
        $policies = WPG_Policy_Engine::instance()->get_active_policies();

        // Sanitize: don't expose internal implementation details.
        return array_map(function ($p) {
            return [
                'id' => $p['id'],
                'name' => $p['name'],
                'description' => $p['description'] ?? '',
                'decision' => $p['decision'] ?? 'deny',
                'abilities' => $p['abilities'] ?? ['*'],
                'priority' => $p['priority'] ?? 100,
                'active' => $p['active'] ?? true,
            ];
        }, $policies);
    }

    /**
     * Handle wpg/agent-profile ability call.
     *
     * @param array $input Unused.
     *
     * @return array The agent profile.
     */
    public function handle_agent_profile(array $input): array
    {
        $agent_info = WPG_Agent_Detector::instance()->detect();

        $user = wp_get_current_user();

        return [
            'agent' => $agent_info['agent'],
            'agent_type' => $agent_info['agent_type'],
            'confidence' => $agent_info['confidence'],
            'session_id' => $agent_info['session_id'],
            'user' => [
                'id' => $user->ID,
                'login' => $user->user_login,
                'roles' => $user->roles,
            ],
            'governance' => [
                'active_policies' => count(WPG_Policy_Engine::instance()->get_active_policies()),
                'gate_status' => get_transient('wpg_governance_paused') ? 'paused' : 'active',
            ],
        ];
    }

    /**
     * Handle wpg/report-generate ability call.
     *
     * @param array $input The report parameters.
     *
     * @return array The generated report.
     */
    public function handle_report_generate(array $input): array
    {
        $period = $input['period'] ?? 'week';

        $stats = WPG_Audit_Log::instance()->get_stats($period);

        $active_policies = WPG_Policy_Engine::instance()->get_active_policies();

        return [
            'report_type' => 'governance_compliance',
            'generated_at' => current_time('mysql', true),
            'period' => $period,
            'site' => get_bloginfo('url'),
            'plugin_version' => WPG_VERSION,
            'summary' => $stats,
            'active_policies' => count($active_policies),
            'policy_list' => array_map(fn($p) => [
        'id' => $p['id'],
        'name' => $p['name'],
        ], $active_policies),
        ];
    }
}
