/**
 * AOS Governance for WP — Admin JavaScript
 *
 * Handles policy CRUD, visual condition builder, ability picker,
 * simulation runner, and settings management.
 *
 * @package aos-governance
 * @since   1.0.0
 */

(function () {
    'use strict';

    const API_BASE = wpgAdmin.restUrl;
    const NONCE = wpgAdmin.nonce;

    // Common field options for the condition builder.
    const FIELD_OPTIONS = [
        { value: 'args.count', label: 'Item count (args.count)', hint: 'Number of items affected' },
        { value: 'args.role', label: 'User role (args.role)', hint: 'e.g., administrator, editor' },
        { value: 'args.post_type', label: 'Post type (args.post_type)', hint: 'e.g., post, page' },
        { value: 'args.post_status', label: 'Post status (args.post_status)', hint: 'e.g., draft, publish' },
        { value: 'args.option_name', label: 'Option name (args.option_name)', hint: 'WordPress option being changed' },
        { value: 'args.file', label: 'File path (args.file)', hint: 'File being accessed' },
        { value: 'args.slug', label: 'Plugin/theme slug (args.slug)', hint: 'Plugin or theme identifier' },
        { value: 'agent', label: 'AI Agent name', hint: 'Name of the AI agent' },
        { value: 'agent_type', label: 'Agent type', hint: 'e.g., mcp, simulation' },
        { value: 'custom', label: '— Custom field —', hint: 'Enter a custom field path' },
    ];

    const OPERATOR_OPTIONS = [
        { value: 'equals', label: 'equals', hint: 'Exact match' },
        { value: 'not_equals', label: 'does not equal', hint: 'Anything except this value' },
        { value: 'greater_than', label: 'is greater than', hint: 'Numeric comparison' },
        { value: 'less_than', label: 'is less than', hint: 'Numeric comparison' },
        { value: 'greater_than_or_equal', label: 'is at least', hint: 'Greater than or equal' },
        { value: 'less_than_or_equal', label: 'is at most', hint: 'Less than or equal' },
        { value: 'contains', label: 'contains text', hint: 'Text includes this substring' },
        { value: 'not_contains', label: 'does not contain', hint: 'Text does not include substring' },
        { value: 'starts_with', label: 'starts with', hint: 'Text begins with this value' },
        { value: 'ends_with', label: 'ends with', hint: 'Text ends with this value' },
        { value: 'in', label: 'is one of', hint: 'Value is in a comma-separated list' },
        { value: 'exists', label: 'exists (any value)', hint: 'Field is present' },
        { value: 'not_exists', label: 'does not exist', hint: 'Field is missing' },
    ];

    /**
     * Make an authenticated REST API request.
     */
    async function apiRequest(endpoint, options = {}) {
        const url = API_BASE + endpoint;
        const config = {
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': NONCE,
            },
            ...options,
        };

        if (config.body && typeof config.body !== 'string') {
            config.body = JSON.stringify(config.body);
        }

        const response = await fetch(url, config);
        return response.json();
    }

    // ===== Policy Toggle =====

    document.querySelectorAll('.wpg-policy-toggle').forEach(toggle => {
        toggle.addEventListener('change', async function () {
            const id = this.dataset.id;
            const active = this.checked;
            const row = this.closest('tr');

            try {
                const result = await apiRequest(`policies/${id}/toggle`, {
                    method: 'POST',
                    body: { active },
                });

                if (result.success) {
                    row.classList.toggle('wpg-row-inactive', !active);
                } else {
                    this.checked = !active;
                    alert('Failed to toggle policy.');
                }
            } catch (err) {
                this.checked = !active;
                console.error('WPG: Toggle failed', err);
            }
        });
    });

    // ===== Visual Condition Builder =====

    let conditionCounter = 0;

    function createConditionRow(data = {}) {
        conditionCounter++;
        const idx = conditionCounter;
        const row = document.createElement('div');
        row.className = 'wpg-condition-row';
        row.dataset.idx = idx;

        // Field select
        let fieldOptions = FIELD_OPTIONS.map(f => {
            const sel = f.value === (data.field || '') ? ' selected' : '';
            return `<option value="${f.value}" title="${f.hint}"${sel}>${f.label}</option>`;
        }).join('');

        // If data.field is custom (not in our list), select "custom"
        const isCustomField = data.field && !FIELD_OPTIONS.find(f => f.value === data.field && f.value !== 'custom');

        if (isCustomField) {
            fieldOptions = FIELD_OPTIONS.map(f => {
                const sel = f.value === 'custom' ? ' selected' : '';
                return `<option value="${f.value}" title="${f.hint}"${sel}>${f.label}</option>`;
            }).join('');
        }

        // Operator select
        const operatorOptions = OPERATOR_OPTIONS.map(o => {
            const sel = o.value === (data.operator || 'equals') ? ' selected' : '';
            return `<option value="${o.value}" title="${o.hint}"${sel}>${o.label}</option>`;
        }).join('');

        const needsValue = !['exists', 'not_exists'].includes(data.operator || 'equals');
        const valueDisplay = needsValue ? '' : 'style="display:none;"';
        const customDisplay = isCustomField ? '' : 'style="display:none;"';

        let valueStr = '';
        if (data.value !== undefined && data.value !== null) {
            valueStr = typeof data.value === 'object' ? JSON.stringify(data.value) : String(data.value);
        }

        row.innerHTML = `
            <div class="wpg-condition-fields">
                <div class="wpg-condition-field-group">
                    <label>If</label>
                    <select class="wpg-cond-field" data-idx="${idx}">
                        ${fieldOptions}
                    </select>
                    <input type="text" class="wpg-cond-custom-field" data-idx="${idx}"
                        placeholder="e.g., args.my_field" value="${isCustomField ? data.field : ''}" ${customDisplay}>
                </div>
                <div class="wpg-condition-field-group">
                    <label>&nbsp;</label>
                    <select class="wpg-cond-operator" data-idx="${idx}">
                        ${operatorOptions}
                    </select>
                </div>
                <div class="wpg-condition-field-group wpg-cond-value-wrap" data-idx="${idx}" ${valueDisplay}>
                    <label>Value</label>
                    <input type="text" class="wpg-cond-value" data-idx="${idx}"
                        placeholder="e.g., 10" value="${valueStr}">
                </div>
                <button type="button" class="button wpg-cond-remove" data-idx="${idx}" title="Remove condition">✕</button>
            </div>
        `;

        // Event: field change
        row.querySelector('.wpg-cond-field').addEventListener('change', function () {
            const customInput = row.querySelector('.wpg-cond-custom-field');
            if (this.value === 'custom') {
                customInput.style.display = '';
                customInput.focus();
            } else {
                customInput.style.display = 'none';
                customInput.value = '';
            }
        });

        // Event: operator change (hide value for exists/not_exists)
        row.querySelector('.wpg-cond-operator').addEventListener('change', function () {
            const valueWrap = row.querySelector('.wpg-cond-value-wrap');
            if (['exists', 'not_exists'].includes(this.value)) {
                valueWrap.style.display = 'none';
            } else {
                valueWrap.style.display = '';
            }
        });

        // Event: remove
        row.querySelector('.wpg-cond-remove').addEventListener('click', function () {
            row.remove();
        });

        return row;
    }

    function getConditionsFromBuilder() {
        const conditions = [];
        document.querySelectorAll('.wpg-condition-row').forEach(row => {
            const fieldSelect = row.querySelector('.wpg-cond-field');
            const customField = row.querySelector('.wpg-cond-custom-field');
            const operator = row.querySelector('.wpg-cond-operator').value;
            const valueInput = row.querySelector('.wpg-cond-value');

            let field = fieldSelect.value;
            if (field === 'custom') {
                field = customField.value.trim();
            }

            if (!field) return;

            const condition = { field, operator };

            if (!['exists', 'not_exists'].includes(operator)) {
                let val = valueInput.value.trim();
                // Auto-detect numeric values
                if (val !== '' && !isNaN(val) && val !== '') {
                    condition.value = Number(val);
                } else if (operator === 'in') {
                    // Split comma-separated values for "in" operator
                    condition.value = val.split(',').map(v => v.trim()).filter(Boolean);
                } else {
                    condition.value = val;
                }
            }

            conditions.push(condition);
        });
        return conditions;
    }

    // Add condition button
    const addCondBtn = document.getElementById('wpg-add-condition');
    if (addCondBtn) {
        addCondBtn.addEventListener('click', () => {
            const list = document.getElementById('wpg-conditions-list');
            list.appendChild(createConditionRow());
        });
    }

    // ===== Ability Picker =====

    function getSelectedAbilities() {
        const checked = [];
        document.querySelectorAll('.wpg-ability-check:checked').forEach(cb => {
            checked.push(cb.value);
        });

        const customInput = document.getElementById('wpg-policy-abilities-custom');
        if (customInput && customInput.value.trim()) {
            const custom = customInput.value.split(',').map(s => s.trim()).filter(Boolean);
            checked.push(...custom);
        }

        return checked;
    }

    function setAbilityCheckboxes(abilities) {
        // Uncheck all first
        document.querySelectorAll('.wpg-ability-check').forEach(cb => {
            cb.checked = false;
        });

        const customAbilities = [];

        abilities.forEach(ability => {
            const checkbox = document.querySelector(`.wpg-ability-check[value="${ability}"]`);
            if (checkbox) {
                checkbox.checked = true;
            } else {
                customAbilities.push(ability);
            }
        });

        const customInput = document.getElementById('wpg-policy-abilities-custom');
        if (customInput) {
            customInput.value = customAbilities.join(', ');
        }
    }

    // ===== Policy Modal =====

    const modal = document.getElementById('wpg-policy-modal');
    const modalForm = document.getElementById('wpg-policy-form');
    const addBtn = document.getElementById('wpg-add-policy');

    function resetModal() {
        if (modalForm) modalForm.reset();
        document.getElementById('wpg-policy-id').value = '';
        document.querySelectorAll('.wpg-ability-check').forEach(cb => { cb.checked = false; });
        const customInput = document.getElementById('wpg-policy-abilities-custom');
        if (customInput) customInput.value = '';
        const condList = document.getElementById('wpg-conditions-list');
        if (condList) condList.innerHTML = '';
        conditionCounter = 0;
    }

    if (addBtn && modal) {
        addBtn.addEventListener('click', () => {
            document.getElementById('wpg-modal-title').textContent = 'Add Custom Policy';
            resetModal();
            modal.style.display = 'flex';
        });
    }

    // Close modal.
    document.querySelectorAll('.wpg-modal-close').forEach(btn => {
        btn.addEventListener('click', () => {
            if (modal) modal.style.display = 'none';
        });
    });

    // Close modal on backdrop click.
    if (modal) {
        modal.addEventListener('click', (e) => {
            if (e.target === modal) modal.style.display = 'none';
        });
    }

    // Edit policy buttons.
    document.querySelectorAll('.wpg-edit-policy').forEach(btn => {
        btn.addEventListener('click', async function () {
            const id = this.dataset.id;

            try {
                const policy = await apiRequest(`policies/${id}`);

                if (policy.error) {
                    alert('Policy not found.');
                    return;
                }

                document.getElementById('wpg-modal-title').textContent = 'Edit Policy';
                resetModal();

                document.getElementById('wpg-policy-id').value = policy.id;
                document.getElementById('wpg-policy-name').value = policy.name || '';
                document.getElementById('wpg-policy-desc').value = policy.description || '';
                document.getElementById('wpg-policy-decision').value = policy.decision || 'deny';
                document.getElementById('wpg-policy-priority').value = policy.priority || 50;
                document.getElementById('wpg-policy-reason').value = policy.reason || '';

                // Set ability checkboxes
                setAbilityCheckboxes(policy.abilities || ['*']);

                // Populate condition builder
                const conditions = policy.conditions || [];
                const condList = document.getElementById('wpg-conditions-list');
                conditions.forEach(cond => {
                    condList.appendChild(createConditionRow(cond));
                });

                modal.style.display = 'flex';
            } catch (err) {
                console.error('WPG: Failed to load policy', err);
            }
        });
    });

    // Save policy form.
    if (modalForm) {
        modalForm.addEventListener('submit', async function (e) {
            e.preventDefault();

            const id = document.getElementById('wpg-policy-id').value;
            const abilities = getSelectedAbilities();
            const conditions = getConditionsFromBuilder();

            if (abilities.length === 0) {
                alert('Please select at least one ability that this policy applies to.');
                return;
            }

            const policy = {
                id: id || undefined,
                name: document.getElementById('wpg-policy-name').value,
                description: document.getElementById('wpg-policy-desc').value,
                decision: document.getElementById('wpg-policy-decision').value,
                priority: parseInt(document.getElementById('wpg-policy-priority').value, 10),
                reason: document.getElementById('wpg-policy-reason').value,
                abilities: abilities,
                conditions: conditions,
                active: true,
            };

            try {
                const method = id ? 'PUT' : 'POST';
                const endpoint = id ? `policies/${id}` : 'policies';

                const result = await apiRequest(endpoint, {
                    method,
                    body: policy,
                });

                if (result.success) {
                    modal.style.display = 'none';
                    location.reload();
                } else {
                    alert('Failed to save policy.');
                }
            } catch (err) {
                console.error('WPG: Save failed', err);
                alert('An error occurred while saving the policy.');
            }
        });
    }

    // Delete policy buttons.
    document.querySelectorAll('.wpg-delete-policy').forEach(btn => {
        btn.addEventListener('click', async function () {
            const id = this.dataset.id;

            if (!confirm('Are you sure you want to delete this policy?')) {
                return;
            }

            try {
                const result = await apiRequest(`policies/${id}`, { method: 'DELETE' });

                if (result.success) {
                    const row = document.querySelector(`tr[data-policy-id="${id}"]`);
                    if (row) row.remove();
                } else {
                    alert('Failed to delete policy.');
                }
            } catch (err) {
                console.error('WPG: Delete failed', err);
            }
        });
    });

    // ===== Settings Form =====

    const settingsForm = document.getElementById('wpg-settings-form');

    if (settingsForm) {
        settingsForm.addEventListener('submit', async function (e) {
            e.preventDefault();

            const settings = {
                deny_first: settingsForm.querySelector('[name="deny_first"]').checked,
                alert_settings: {
                    email_enabled: settingsForm.querySelector('[name="email_enabled"]')?.checked || false,
                    email_to: settingsForm.querySelector('[name="email_to"]')?.value || '',
                    webhook_enabled: settingsForm.querySelector('[name="webhook_enabled"]')?.checked || false,
                    webhook_url: settingsForm.querySelector('[name="webhook_url"]')?.value || '',
                },
            };

            try {
                const result = await apiRequest('settings', {
                    method: 'PUT',
                    body: settings,
                });

                if (result.success) {
                    const btn = settingsForm.querySelector('[type="submit"]');
                    const originalText = btn.textContent;
                    btn.textContent = '✓ Saved!';
                    btn.disabled = true;

                    setTimeout(() => {
                        btn.textContent = originalText;
                        btn.disabled = false;
                    }, 2000);
                }
            } catch (err) {
                console.error('WPG: Settings save failed', err);
                alert('Failed to save settings.');
            }
        });
    }

    // ===== Test Policy (Dry Run) =====

    const testBtn = document.getElementById('wpg-test-policy');

    if (testBtn) {
        testBtn.addEventListener('click', async function () {
            const ability = prompt('Enter an ability name to test (e.g., core/delete-posts):');
            if (!ability) return;

            try {
                const result = await apiRequest('check', {
                    method: 'POST',
                    body: {
                        ability: ability,
                        agent: 'admin-test',
                        agent_type: 'manual',
                    },
                });

                const decision = result.decision === 'allow' ? '✅ ALLOWED' : '🛑 DENIED';
                let msg = `${decision}\n\nAbility: ${ability}`;

                if (result.policy_name) {
                    msg += `\nPolicy: ${result.policy_name}`;
                }
                if (result.reason) {
                    msg += `\nReason: ${result.reason}`;
                }
                msg += `\nEvaluation Time: ${result.elapsed_ms}ms`;

                alert(msg);
            } catch (err) {
                console.error('WPG: Test failed', err);
                alert('Test failed. Check the console for details.');
            }
        });
    }

    // ===== Simulation Runner (Dashboard) =====

    const simBtn = document.getElementById('wpg-run-simulation');
    const simStatus = document.getElementById('wpg-sim-status');

    if (simBtn) {
        simBtn.addEventListener('click', async function () {
            simBtn.disabled = true;
            simBtn.textContent = '⏳ Running 8 scenarios...';
            if (simStatus) simStatus.textContent = '';

            try {
                const result = await apiRequest('simulate/batch', {
                    method: 'POST',
                    body: {},
                });

                if (result.success) {
                    let summary = `✅ Simulation complete!\n\n`;
                    summary += `Total: ${result.total} scenarios\n`;
                    summary += `🛑 Denied: ${result.denied}\n`;
                    summary += `✅ Allowed: ${result.allowed}\n\n`;
                    summary += `--- Results ---\n`;

                    result.results.forEach(r => {
                        const icon = r.decision === 'deny' ? '🛑' : '✅';
                        summary += `${icon} ${r.label}\n   Agent: ${r.agent} | Ability: ${r.ability}\n   Policy: ${r.policy} | ${r.time_ms}ms\n\n`;
                    });

                    alert(summary);

                    if (simStatus) {
                        simStatus.textContent = `Done — ${result.denied} denied, ${result.allowed} allowed. Reloading dashboard...`;
                    }

                    // Reload to see updated stats.
                    setTimeout(() => location.reload(), 1500);
                } else {
                    alert('Simulation failed. Check the console.');
                }
            } catch (err) {
                console.error('WPG: Simulation failed', err);
                alert('Simulation error: ' + err.message);
            }

            simBtn.disabled = false;
            simBtn.textContent = '🧪 Run Test Scenarios';
        });
    }

})();
